/**
 * @file
 * JavaScript behaviors for element #states.
 */

(function ($, Drupal, drupalSettings) {

  'use strict';

  var isChrome = (/chrom(e|ium)/.test(window.navigator.userAgent.toLowerCase()));

  /**
   * Element #states builder.
   *
   * @type {Drupal~behavior}
   */
  Drupal.behaviors.webformElementStates = {
    attach: function (context) {
      $(context).find('.webform-states-table--condition').once('webform-element-states-condition').each(function () {
        var $condition = $(this);
        var $selector = $condition.find('.webform-states-table--selector select');
        var $value = $condition.find('.webform-states-table--value input');
        var $trigger = $condition.find('.webform-states-table--trigger select');

        // Initialize autocompletion.
        $value.autocomplete({minLength: 0}).on('focus', function () {
          $value.autocomplete('search', '');
        });

        // Initialize trigger and selector.
        $trigger.on('change', function () {$selector.change();});

        $selector.on('change', function () {
          var selector = $selector.val();
          var sourceKey = drupalSettings.webformElementStates.selectors[selector];
          var source = drupalSettings.webformElementStates.sources[sourceKey];
          var notPattern = ($trigger.val().indexOf('pattern') === -1);
          if (source && notPattern) {
            // Enable autocompletion.
            $value
              .autocomplete('option', 'source', source)
              .addClass('form-autocomplete');
          }
          else {
            // Disable autocompletion.
            $value
              .autocomplete('option', 'source', [])
              .removeClass('form-autocomplete');
          }
          // Always disable browser auto completion.
          $value.attr('autocomplete', (isChrome ? 'chrome-off-' + Math.floor(Math.random() * 100000000) : 'off'));
        }).change();
      });

      // If the states:state is required or optional the required checkbox
      // should be checked and disabled.
      var $state = $(context).find('.webform-states-table--state select');
      if ($state.length) {
        $state.once('webform-element-states-state')
          .on('change', toggleRequiredCheckbox);
        toggleRequiredCheckbox();
      }
    }
  };

  /**
   * Track required checked state.
   *
   * @type {null|boolean}
   */
  var requiredChecked = null;

  /**
   * Toggle the required checkbox when states:state is required or optional.
   */
  function toggleRequiredCheckbox() {
    var $input = $('input[name="properties[required]"]');
    if (!$input.length) {
      return;
    }

    // Determine if any states:state is required or optional.
    var required = false;
    $('.webform-states-table--state select').each(function () {
      var value = $(this).val();
      if (value === 'required' || value === 'optional') {
        required = true;
      }
    });

    if (required) {
      requiredChecked = $input.prop('checked');
      $input.attr('disabled', true);
      $input.prop('checked', true);
    }
    else {
      $input.attr('disabled', false);
      if (requiredChecked !== null) {
        $input.prop('checked', requiredChecked);
        requiredChecked = null;
      }
    }
    $input.change();
  }

})(jQuery, Drupal, drupalSettings);
